﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Microsoft.CodeAnalysis.CSharp;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [AllowAnonymous]
    [Route("api/feedback")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class FeedbackController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly IFeedbackService feedbackServices;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// The resource services.
        /// </summary>

        private readonly IResourceService resourceServices;
        /// <inheritdoc />
        public FeedbackController(IFeedbackService feedbackServices, IAuditLogService auditLogServices, IResourceService resourceServices)
        {
            this.feedbackServices = feedbackServices;
            this.auditLogServices = auditLogServices;
            this.resourceServices = resourceServices;
        }


        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] FeedbackModel model)
        {
            model = (FeedbackModel)EmptyFilter.Handler(model);
            var resources = await this.resourceServices.PatientAsync(model);
            foreach (var item in resources)
            {
                model.PatientId = item.PatientId;
                model.Mobile = item.Mobile;
            }
            var response = await this.feedbackServices.AddAsync(model);

            return this.Success("Patient feedback has been added successfully.");
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<FeedbackModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] FeedbackFilterModel model)
        {
            model = (FeedbackFilterModel)EmptyFilter.Handler(model);
            var feedbacks = await this.feedbackServices.FetchAsync(model);
            return feedbacks == null ? this.ServerError() : this.Success(feedbacks);
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch-question")]
        [ProducesResponseType(typeof(List<FeedbackModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchQuestionAsync([FromBody] FeedbackFilterModel model)
        {
            model = (FeedbackFilterModel)EmptyFilter.Handler(model);
            var feedbacks = await this.feedbackServices.FetchQuestionAsync(model);
            return feedbacks == null ? this.ServerError() : this.Success(feedbacks);
        }

        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add-question")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddFeedbackQuestionAsync([FromBody] FeedbackModel model)
        {
            model = (FeedbackModel)EmptyFilter.Handler(model);
            var resources = await this.feedbackServices.AddFeedbackQuestionAsync(model);
            switch (resources)
            {
                case -1:
                    return this.Conflict("Given feedback question has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            return this.Success("Feedback question has been added successfully.");
        }



        /// <summary>
        /// The update coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon updated successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] FeedbackModel model)
        {
            model = (FeedbackModel)EmptyFilter.Handler(model);
            var response = await this.feedbackServices.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given question has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            //var auditLogModel = new AuditLogModel
            //{
            //    AccountId = model.ModifiedBy,
            //    LogTypeId = (int)LogTypes.Coupons,
            //    LogFrom = (int)AccountType.Administrator,
            //    LogDate = DateTime.UtcNow,
            //    LogDescription = $"{model.CouponCode} coupon has been updated."
            //};
            // await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Question has been updated successfully.");


        }


        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] FeedbackModel model)
        {
            try
            {
                model = (FeedbackModel)EmptyFilter.Handler(model);
                var response = await this.feedbackServices.DeleteAsync(model.QuestionFeedbackId);
                if (response == 0)
                {
                    return this.ServerError();
                }

                //var auditLogModel = new AuditLogModel
                //{
                //    AccountId = model.ModifiedBy,
                //    LogTypeId = (int)LogTypes.Specializations,
                //    LogFrom = (int)AccountType.Administrator,
                //    LogDate = DateTime.UtcNow,
                //    LogDescription = $"{model.BedNumber} specialization has been deleted."
                //};
                //await this.auditLogServices.LogAsync(auditLogModel);

                return this.Success("Question has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("Question can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }
    }
}